#ifndef __CMidiFile__
#define __CMidiFile__

//	===========================================================================

#include <Collections/TCountedPointerArray.hpp>
#include <Basics/CSystemString.hpp>
#include <IO/CFileStream.hpp>
#include "../Audio/TAudioBuffer.hpp"
#include "CMidiEvent.hpp"
#include "CMidiSequence.hpp"
using Exponent::Midi::CMidiSequence;
using Exponent::Midi::CMidiEvent;
using Exponent::Audio::TAudioBuffer;
using Exponent::IO::CFileStream;
using Exponent::Basics::CSystemString;
using Exponent::Collections::TCountedPointerArray;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidiFile CMidiFile.hpp
		 * @brief Provides SMF i/o
		 *
		 * Midi file io, currently only supports types 0 and 1 (who the hell uses type 2 midi files :S??)\n
		 * All read operations are carried out as follows:\n
		 * @code
		 * // First we check that its a valid midi file
		 * if (!CMidiFile::isValidFormat("Path/To/Midi/File.mid"))
		 * {
		 *		cout << "This is not a midi file" << endl;
		 *		return false;
		 * }
		 * 
		 * // Obviously a valid file, we can create the midi file
		 * CMidiFile file;
		 *
		 * // Open
		 * if (!file.openFile(CFileStream::e_input, "Path/To/Midi/File.mid"))
		 * {
		 *		cout << "Failed to open the midi file" << endl;
		 *		return false;
		 * }
		 *
		 * // Midi files are read in to midi sequences
		 * CMidiSequence sequence(sampleRate);
		 * 
		 * // Now read the file in to the sequence
		 * if (!file.readFile(sequence))
		 * {
		 *		cout << "Failed to read the midi file sequence" << endl;
		 *		return false;
		 * }
		 *
		 * // The sequence is valid if we go to here, now you can use it for anything you want....
		 *
		 * @endcode
		 * \n
		 * All write operations are carried out as follows:\n
		 * @code
		 * // Setup the sequence
		 * CMidiSequence sequence;
		 *
		 * // Add your events
		 * ...
		 *
		 * // Now write
		 * CMidiFile file;
		 * 
		 * // Open
		 * if (!file.openFile(CFileStream::e_output, "Path/To/Midi/File.mid"))
		 * {
		 *		cout << "Failed to open file to write" << endl;
		 *		return false;
		 * }
		 *
		 * // Now we can write the file
		 * if (!file.writeFile(sequence))
		 * {
		 *		cout << "Failed to write file" << endl;
		 *		return false;
		 * }
		 * @endcode
		 *
		 * @see CMidiSequence
		 * @see CMidiTrack
		 * 
		 * @date 23/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * @note Documented online at <A HREF="http://www.borg.com/~jglatt/tech/midifile.htm">http://www.borg.com/~jglatt/tech/midifile.htm</A>
		 *
		 * $Id: CMidiFile.hpp,v 1.3 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidiFile : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 */
			CMidiFile();

			/**
			 * Destruction
			 */
			virtual ~CMidiFile();

//	===========================================================================

			/**
			 * Open the file
			 * @param mode The mode of opening, read or write
			 * @param filename The name of the file to open
			 * @retval bool True if stream is opened
			 */
			bool openFile(CFileStream::EStreamMode mode, const CSystemString &filename);

			/**
			 * Close the file
			 * @retval bool True if closed properly, false otherwise
			 */
			bool closeFile();

//	===========================================================================

			/**
			 * Read the file
			 * @param sequence The sequence to fill in
			 * @retval bool True if sequence is filled properly, false on error
			 */
			bool readFile(CMidiSequence &sequence);

			/**
			 * WRite the file
			 * @param sequence The sequence to write, expected to be in sorted order
			 * @retval bool True if sequence is written properly, false otherwise
			 */
			bool writeFile(const CMidiSequence &sequence);

//	===========================================================================

			/**
			 * Get the extension of the file format, each array entry sould contain one string
			 * @retval CString The extension of a midi file
			 */
			static CString getFileExtension();

			/**
			 * Check if a file is a valid format of this file format
			 * @param filename The name of the file to load
			 * @retval bool True if the file format matches (checks actual binary data, not just extension
			 */
			static bool isValidFormat(const CSystemString &filename);

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * @struct SMTHDChunk CMidiFile.hpp
			 * @brief Chunk header for midi files
			 */
			struct SMTHDChunk
			{
				char m_mthdChunk[4];				/**< 'MThd' */
				unsigned long m_mthdSize;			/**< Size of the mthd chunk */

				unsigned short m_fileType;			/**< What format of midi file */
				unsigned short m_numberOfTracks;	/**< Total number of tracks */
				unsigned short m_timeFormat;		/**< SMPTE or Tempo based ( < 0 == SMPTE) */

				double m_samplesPerTick;			/**< Total samples per tick */
			};

			/**
			 * @struct SMTRKChunk CMidiFile.hpp
			 * @brief Track header for midi files
			 */
			struct SMTRKChunk
			{
				/* Here's the 8 byte header that all chunks must have */
				char m_mtrkChunk[4];				/**< 'MTrk' */
				unsigned long m_mtrkSize;			/**< Size of data block */
			};

			/**
			 * @struct SMidiEvent CMidiFile.hpp
			 * @brief MidiEvent header
			 */
			struct SMidiEvent
			{
				unsigned long m_delaySinceLast;		/**< Delay since the last event */
				unsigned char m_statusByte;			/**< Status informaiton */
				unsigned char m_midiData[2];		/**< The midi data */
			};

//	===========================================================================
//			READ FUNCTIONS
//	===========================================================================

			/**
			 * Is this a valid midi file
			 * @param stream The stream to read
			 * @param chunk The chunk to fill in
			 * @retval bool True if midi file format
			 */
			static bool readFileHeader(CFileStream &stream, SMTHDChunk &chunk);

			/**
			 * Read the tracks
			 * @param stream The stream to read
			 * @param headChunk The head chunk to fill in
			 * @param sequence The sequence to fill in
			 * @retval bool True if read correctly, false otherwise
			 */
			static bool readTrackChunks(CFileStream &stream, SMTHDChunk &headChunk, CMidiSequence &sequence);

			/**
			 * Write a variable length unsigned long
			 * @param sizeRead On return holds the size of the data read in
			 * @param stream The stream to write to
			 * @retval unsigned long The variable length unsigned long as read from disk
			 */
			static unsigned long readVariableLengthUnsignedLong(CFileStream &stream, int &sizeRead);

//	===========================================================================
//			WRITE FUNCTIONS
//	===========================================================================

			/**
			 * Write the header
			 * @param stream The stream to write
			 * @param sequence The sequence to write from
			 * @retval bool True if wrote header properly
			 */
			static bool writeFileHeader(CFileStream &stream, const CMidiSequence &sequence);

			/**
			 * Write the meta track
			 * @param stream The stream to write
			 * @param sequence The sequence to write from
			 */
			static void writeMetaTrack(CFileStream &stream, const CMidiSequence &sequence);

			/**
			 * Write the tempo in as a meta event
			 * @param stream The stream to read
			 * @param ts The time signature
			 */
			static void writeTimeSignature(CFileStream &stream, const CTimeSignature ts);

			/**
			 * Write the tracks
			 * @param stream The stream to read
			 * @param track The track to write
			 * @param ticksPerSample The number of ticks per sample @see computeTicksPerSample
			 * @retval bool True if written correctly, false otherwise
			 */
			static bool writeTrack(CFileStream &stream, const CMidiTrack &track, const double ticksPerSample);

			/**
			 * Write the name of a track
			 * @param name THe name of the track
			 * @param stream The stream to read
			 */
			static void writeTrackName(CFileStream &stream, const CString &name);

			/**
			 * Write the midi channel of a track
			 * @param channel The midi channel
			 * @param stream The stream to read
			 */
			static void writeTrackMidiChannel(CFileStream &stream, const long channel);

			/**
			 * Write the end of a track
			 * @param stream The stream to read
			 */
			static void writeTrackEnd(CFileStream &stream);

			/**
			 * Write the tempo in as a meta event
			 * @param stream The stream to read
			 * @param bpm The bpm
			 */
			static void writeTempo(CFileStream &stream, const double bpm);

			/**
			 * Write a variable length unsigned long
			 * @param value The value to write
			 * @param stream The stream to write to
			 */
			static void writeVariableLengthUnsignedLong(CFileStream &stream, const unsigned long value);

			/**
			 * Is a writable event
			 * @param event The event to consider
			 * @retval bool True if writable event, false otherwise
			 */
			static bool isWritableEvent(const CMidiEvent &event);

//	===========================================================================
//			UTILITY FUNCTIONS
//	===========================================================================

			/**
			 * Is actually an MTHD chunk
			 * @param chunk The chunk to check
			 * @retval bool True if MTHD chunk
			 */
			static bool isValidMTHDChunk(const SMTHDChunk &chunk);

			/**
			 * Is actually an MTRK chunk
			 * @param chunk The chunk to check
			 * @retval bool True if MTRK chunk
			 */
			static bool isValidMTRKChunk(const SMTRKChunk &chunk);

			/**
			 * Compute teh samples per tick
			 * @param headChunk The head chunk to fill in
			 * @param sampleRate The sample rate of the sequence
			 * @param bpm The bpm of the track, ignored if SMPTE timing being used
			 */
			static void computeSamplesPerTick(SMTHDChunk &headChunk, const double sampleRate, const double bpm);

			/**
			 * Compute the number of ticks per sample
			 * @param sampleRate The samplerate of the sequence
			 * @param bpm The bpm of the sequence
			 * @retval double The number of ticks every sample
			 */
			static double computeTicksPerSample(const double sampleRate, const double bpm);

			/**
			 * Post process a midi track
			 * @param headChunk The head chunk to get information from
			 * @param sequence The sequence to process
			 */
			static void processMidiSequence(SMTHDChunk &headChunk, CMidiSequence &sequence);

//	===========================================================================

			const static char CMIDI_FILE_EVENT_SIZES[];								/**< Size of all the default events. Dont even try to understand this, it works ok ;) */
			const static unsigned short CMIDI_FILE_PPQ = 960;						/**< PPQ size for output files */

//	===========================================================================
			
			CFileStream m_stream;													/**< The file stream that we have */
		};
	}
}
#endif	// End of CMidiFile.hpp